package ie.ucd.sixth.core.cyber.utils.html;

import java.util.HashMap;
import java.util.Set;
import java.util.logging.Logger;

/*
 * convert a template that has commands into the full template
 * 
 * sample template (brackets not included):
 * 	var:%i=0to47; (template contains a variable %i whose value starts at 0)
	var:%j=0; (template contains a variable %j whose value starts at 0)
	repeat template using:%i++; (template is to be repeated until i has value 47 (dictated by max value) increasing i by 1 each time)
	begin template; (this indicates that all following lines are part of the template)
	<row%icol%j>  (%i and %j are place holders for the values of i and j)
	<Station>row%icol%j+1</Station> (%j+1 means that the value to be used here is the value of j +1)
	<Current_Condition>row%icol%j+2</Current_Condition>
	<Temperature_Degrees_Centegrade>row%icol%j+3</Temperature_Degrees_Centegrade>
	<Dew_Point_Degrees_Centegrade>row%icol%j+4</Dew_Point_Degrees_Centegrade>
	<Humidity_Percentage>row%icol%j+5</Humidity_Percentage>
	<Wind_Average_KMH>row%icol%j+6</Wind_Average_KMH>
	<Wind_Gust_KMH>row%icol%j+7</Wind_Gust_KMH>
	<Rain_MM>row%icol%j+8</Rain_MM>
	<Barometer_MB>row%icol%j+9</Barometer_MB>
	<Barometer_Trend>row%icol%j+10</Barometer_Trend>
	<TimeStamp>row%icol%j+11</TimeStamp>
</row%icol%j>
 */
public class TemplateTranslator {
	private static Logger logger = Logger.getLogger(Logger.GLOBAL_LOGGER_NAME);

	//language that should be used for instructions in template
	private final String plus = "+";
	private final String minus = "-";
	private final String incrementByOne = "++";
	private final String decrementByOne = "--";
	private final String variableDeclaration = "var:";
	private final String repeatInstruction = "repeat template using:";
	private final String beginTemplateInstruction = "begin template";
	//variables for compiling full template
	private String fullTemplate = "";
	String[] templateArray;
	private boolean templateOK = false;
	private HashMap<String, int[]> variablesAndValues; //inital values and maximum values
	private HashMap<String, String> variablesAndValueChanges; //variable changes
	private String errorString = "";


	public TemplateTranslator(){

	}


	/*
	 * this method attempts to compile the template based on the instructions before it
	 * any errors in the instructions should cause the method to return false
	 * letting the caller method know that there is a problem with the template
	 * and it needs to be fixed
	 */
	public boolean translateTemplate(String template){
		templateArray = template.split(";");
		/*
		 * first we need to check what instructions and variables are being used (if any)
		 * if the first line of the template is "begin template;" we know nothing needs to be done
		 * and we can return the template as it is (without the begin template; instruction
		 */
		if(templateArray.length == 2 && templateArray[0].equalsIgnoreCase("begin template;")){
			fullTemplate = templateArray[1];
			templateOK = true; 
			return true;
		}else{
			variablesAndValues = new HashMap<String, int[]>();
			variablesAndValueChanges = new HashMap<String, String>();
		}

		/*
		 * try to declare variables
		 * if it doesnt work we return false
		 */
		if(!declareVariables()){ 
			logger.severe("failed on variables");
			return false; 
		}

		/*
		 * just for testing: variables, values and max values
		 */
//		Set<String> varSet = variablesAndValues.keySet();
//		for (String string : varSet) {
//			int[] value = variablesAndValues.get(string);
//
//			logger.info("variablesAndValues.... key, value pair: " +string+", "+value[0] +" AND "+value[1]);
//		}

		/*
		 * check what repeat instructions have been given
		 */
		if(!defineInstructions()){
			logger.severe("failed on instructions");
			return false;
		}

		/*
		 * just for testing: variables and value changes
		 */
//
//		Set<String> varChangeSet = variablesAndValueChanges.keySet();
//		for (String string : varChangeSet) {
//			String changes = variablesAndValueChanges.get(string);
//		}

		/*
		 * check the begin template instruction comes before the start of a template
		 */
		if(!confirmBeginTemplateInstruction()){
			logger.severe("failed finding begin");
			return false;
		}
		if(!compileFullTemplate()){
			logger.severe("failed compiling");
			return false;
		}

		fullTemplate = performInnerTemplateInstruction(fullTemplate.trim());


		logger.info("return true: no fails");

		return true;
	}

	private boolean compileFullTemplate(){
		String templateString = templateArray[templateArray.length-1];

		//get the full list of variables:
		Set<String> variableSet = variablesAndValues.keySet();

		//get the list of variables that need to be changed:
		Set<String> variablesToBeChangedSet = variablesAndValueChanges.keySet();
		int i =0;
		boolean variablesInBounds = true;
		while(variablesInBounds){

			String tempTemplate = templateString;

			for(String string : variableSet){
				if(variablesAndValueChanges.containsKey(string)){ //we need to follow instructions provided
//					logger.info("have variable requiring change: " +string);
					int[] tempArray = variablesAndValues.get(string);
//					for(int j =0; i<tempArray.length; i++){
//						logger.info("value at: " +j+"= " +tempArray[j]);
//					}
					int currentValue = tempArray[0];
//					logger.info("replacing in tempTemplate: "+string+" becomes: "  +currentValue);
					String newString = Integer.toString(currentValue);
					tempTemplate = tempTemplate.replaceAll(string, newString); //replace the placeholders

					/*
					 * update the changes specified by template instructions
					 */
					String change = variablesAndValueChanges.get(string);
					if(change.equalsIgnoreCase(incrementByOne)){
						currentValue++;
						setCurrentValue(string, currentValue, variablesAndValues.get(string)[1]);
					}else if(change.equalsIgnoreCase(decrementByOne)){
						currentValue--;
						setCurrentValue(string, currentValue, variablesAndValues.get(string)[1]);
					}else{ //we need to convert this change to number format
						try{
							int changeInt = Integer.parseInt(change);
							currentValue+=changeInt;
							setCurrentValue(string, currentValue, variablesAndValues.get(string)[1]);
						}catch(NumberFormatException e){
							logger.severe("PROBLEM CONVERTING NUMBER... PROBLEM WITH TEMPLATE TRANSLATOR AND NOT USER PROVIDED TEMPLATE");
							logger.severe(e+"");
							return false;
						}
					}
					//					tempTemplate = performInnerTemplateInstruction(tempTemplate);
					//					fullTemplate+=tempTemplate;


				}else{ // we just need to replace
//					logger.info("have variable just need to replace: " +string);
					int[] tempArray = variablesAndValues.get(string);
//					for(int j =0; i<tempArray.length; i++){
//						logger.info("value at: " +j+"= " +tempArray[j]);
//					}
					int currentValue = tempArray[0];
					//					logger.info("replacing in tempTemplate: "+string+" becomes: "  +currentValue);
					String newString = Integer.toString(currentValue);
					tempTemplate = tempTemplate.replaceAll(string, newString); //replace the placeholders
					//					tempTemplate = performInnerTemplateInstruction(tempTemplate);
					//					logger.info(tempTemplate);
				}

			}
			fullTemplate+=tempTemplate;

			//check each variable is still in bounds
			for (String string : variableSet) {
				int curr = variablesAndValues.get(string)[0];
				int max = variablesAndValues.get(string)[1]; //this holds the maximum value
				if(curr>max){ //if the index has exceeded the maximum for any variable we need to stop
					variablesInBounds = false;
				}
			}
		}

		//		fullTemplate = performInnerTemplateInstruction(fullTemplate);
		return true; //if we get this far we know everything is ok
	}

	private String performInnerTemplateInstruction(String templateToBeChanged){
		//		logger.info("template to be changed: " +templateToBeChanged);
		String tempTemplate = templateToBeChanged;
		String[] tempVariable;
		if(templateToBeChanged.contains(incrementByOne)){
//			logger.info("found incremement by 1: " +templateToBeChanged);
			tempVariable = templateToBeChanged.split(incrementByOne); //now we know that the last of each array is a number that needs to be incremented
			for(int i = 0; i<tempVariable.length-1; i++) {
				String firstString = tempVariable[i];
				String tempFirstNumber = "";

				for(int j=firstString.length(); j>=0; j--){
					Character c = firstString.charAt(j);
					if(Character.isDigit(c)){
						tempFirstNumber = tempFirstNumber+c;
					}else{
						break;
					}
				}




				//increment the number
				try{
					int firstInt = Integer.parseInt(tempFirstNumber);

					int result = firstInt++;

					//replace all occurance of this instruction with new number
					String stringToReplace = firstInt+incrementByOne;
					tempTemplate = tempTemplate.replaceAll(stringToReplace, Integer.toString(result));
				}catch(NumberFormatException e){
					logger.severe("PROBLEM WITH INTERNAL TEMPLATE INSTRUCTIONS... CANNOT INCREMENT NUMBER");
					logger.severe(e+"");
				}

			}

		}if(templateToBeChanged.contains(decrementByOne)){
//			logger.info("found decrement by 1: "+ templateToBeChanged);
			tempVariable = templateToBeChanged.split(decrementByOne); //now we know that the last of each item in array needs to be incremented
			for(int i = 0; i<tempVariable.length-1; i++) {
				String firstString = tempVariable[i];
				String tempFirstNumber = "";

				for(int j=firstString.length(); j>=0; j--){
					Character c = firstString.charAt(j);
					if(Character.isDigit(c)){
						tempFirstNumber = tempFirstNumber+c;
					}else{
						break;
					}
				}




				//perform addition of the two numbers
				try{
					int firstInt = Integer.parseInt(tempFirstNumber);
					int result = firstInt--;

					//replace all occurance of this instruction with new number
					String stringToReplace = firstInt+decrementByOne;
					tempTemplate = tempTemplate.replaceAll(stringToReplace, Integer.toString(result));
				}catch(NumberFormatException e){
					logger.severe("PROBLEM WITH INTERNAL TEMPLATE INSTRUCTIONS... CANNOT DECREMENT NUMBER");
					logger.severe(e+"");
				}

			}

		}	if(templateToBeChanged.contains(plus)){
//			logger.info("found plus: " +templateToBeChanged);
			tempVariable = templateToBeChanged.split("\\+"); //now we know that the last of each item in array should be added to the first in the next..
			for(int i = 0; i<tempVariable.length-1; i++) {
				String firstString = tempVariable[i];
//				logger.info("firstString: " +firstString);
				String tempFirstNumber = "";

				for(int j=firstString.length()-1; j>=0; j--){
					Character c = firstString.charAt(j);
					if(Character.isDigit(c)){
						tempFirstNumber = tempFirstNumber+c;
					}else{
						break;
					}
				}


				String secondString = tempVariable[i+1];
//				logger.info("second string: " +secondString);
				String tempSecondNumber = "";
//				logger.info("converting Strings: " +firstString +" and " +secondString);
				for(int j=0; j<secondString.length(); j++){
					Character c = secondString.charAt(j);
					if(Character.isDigit(c)){
						tempSecondNumber = c+tempSecondNumber;
					}else{
						break;
					}
				}


				//perform addition of the two numbers
				try{
//					logger.info("before converting to ints: " +tempFirstNumber+" and " +tempSecondNumber);
					int firstInt = Integer.parseInt(tempFirstNumber);
					int secondInt = Integer.parseInt(tempSecondNumber);
					int result = firstInt+secondInt;
//					logger.info("result: "+firstInt + "+"+secondInt+"="+result);

					//replace all occurance of this instruction with new number
					String stringToReplace = firstInt+"\\"+plus+secondInt;
//					logger.info("replacing: " +stringToReplace +" with: " +result + " in String: " +tempTemplate);
					tempTemplate = tempTemplate.replaceAll(stringToReplace, Integer.toString(result));
//					logger.info("template after replace: " +tempTemplate);
				}catch(NumberFormatException e){
					logger.severe("PROBLEM WITH INTERNAL TEMPLATE INSTRUCTIONS... CANNOT ADD NUMBERS: " +tempFirstNumber+" and " +tempSecondNumber );
					logger.severe(e+"");
				}

			}



		}if(templateToBeChanged.contains(minus)){
//			logger.info("found minus: " +templateToBeChanged );
			tempVariable = templateToBeChanged.split(minus); //now we know that the last of each item in array should be added to the first in the next..
			for(int i = 0; i<tempVariable.length-1; i++) {
				String firstString = tempVariable[i];
				String tempFirstNumber = "";

				for(int j=firstString.length(); j>=0; j--){
					Character c = firstString.charAt(j);
					if(Character.isDigit(c)){
						tempFirstNumber = tempFirstNumber+c;
					}else{
						break;
					}
				}


				String secondString = tempVariable[i+1];

				String tempSecondNumber = "";

				for(int j=0; j<secondString.length(); j++){
					Character c = secondString.charAt(j);
					if(Character.isDigit(c)){
						tempSecondNumber = c+tempSecondNumber;
					}else{
						break;
					}
				}

				//perform addition of the two numbers
				try{
					int firstInt = Integer.parseInt(tempFirstNumber);
					int secondInt = Integer.parseInt(tempSecondNumber);
					int result = firstInt-secondInt;

					//replace all occurance of this instruction with new number
					String stringToReplace = firstInt+minus+secondInt;
					tempTemplate = tempTemplate.replaceAll(stringToReplace, Integer.toString(result));
				}catch(NumberFormatException e){
					logger.severe("PROBLEM WITH INTERNAL TEMPLATE INSTRUCTIONS... CANNOT SUBTRACT NUMBERS");
					logger.severe(e+"");
				}

			}

		}


//		logger.info("returning template: " +tempTemplate);
		return tempTemplate;
	}


	private boolean declareVariables(){

		/*
		 * declare variables  eg:
		 * var:%i=0to47; 
		 * var:%j=0;
		 */
		for(int i = 0; i< templateArray.length; i++){ 
			String instruction = templateArray[i];
			instruction = instruction.trim();
//			logger.info("checking if "+instruction+" starts with " +variableDeclaration);
			if(instruction.startsWith(variableDeclaration)){ // this line contains a variable initiliazation eg: var:i=0;
//				logger.info("it does");
				String[] var = instruction.split(":"); //var[1] should hold the variable and its starting value (eg i=0)
				if(var.length == 2){ //then this is ok
					String[] varInit = var[1].split("=");
					if(varInit.length == 2){ //var[0] should hold variable name and var[1] should hold numerical starting value
						String varName = varInit[0];
						try{
							if(varInit[1].contains("to")){ //a maximum value for this variable has been specified
								String[] startAndMaxVals = new String[2]; //we need an array to store the start and max value for each variable
								startAndMaxVals = varInit[1].split("to");
								if(startAndMaxVals.length==2){ //this works 
									try{
										int startValue = Integer.parseInt(startAndMaxVals[0]); //if the number can be resolved great, if not template broken
										int maxValue = Integer.parseInt(startAndMaxVals[1]);
										int[] startAndMaxInts = {startValue, maxValue};
										variablesAndValues.put(varName, startAndMaxInts);
									}catch(NumberFormatException e){
										logger.severe("String could not be converted to number for template: " +e);
										return false;
									}
								}
							}else{ //no maximum value specified
								int value = Integer.parseInt(varInit[1]); //if the number can be resolved great, if not template broken
								int[] startAndMaxInts = {value, value}; //if no max has been specified the max is the original value
								variablesAndValues.put(varName, startAndMaxInts);
							}

						}catch(NumberFormatException e){
							logger.severe("String could not be converted to number for template: " +e);
							return false;
						}


					}else{ //variable has no value assigned - template needs to be fixed

						return false;
					}

				}else{ //variable not declared and init properly .. template needs to be fixed
					return false;
				}

			}else{ //no variables to be initialized so we need to return false
				/*
				 * we only want to declare variables in this loop
				 * all variable inits should come first so when we find something else we can break out
				 */
//				logger.info("it doesnt");
				break;

			}
		}

		return true; //if we make it this far we know nothing went wrong
	}

	private boolean defineInstructions(){
		boolean hasInstruction = false;
//		logger.info("defining instructions");
//		logger.info("size of variablesAndValues: " +variablesAndValues.size());
//		logger.info("size of templateArray: " +templateArray.length);
		for(int i=variablesAndValues.size(); i<templateArray.length; i++){ //starting at the number of variables so we dont iterate them again
//			logger.info("in for loop... " +i);
			String instruction = templateArray[i].trim();
			if(instruction.startsWith(repeatInstruction)){ //found instruction
				String changes_ = templateArray[i].trim();
				String[] changes = changes_.split(":");
				if(changes.length == 2){ //correct format, changes[1] should hold instruction
					String changeString = changes[1];
					//need to make sure changeString contains the name of a variable
					Set<String> variableSet = variablesAndValues.keySet();
					for (String string : variableSet) {

//						logger.info("changestring: " +changeString);
//						logger.info("checking string: " +string);

						if(changeString.equals(string+incrementByOne)){

							variablesAndValueChanges.put(string, incrementByOne);

						}else if(changeString.equals(string+decrementByOne)){

							variablesAndValueChanges.put(string, decrementByOne);

						}else if(changeString.startsWith(string+"=")){ //we need to find out what is on the right hand side
							String[] equalsChanges = changeString.split("=");
							if(equalsChanges.length==2){ //correct format
								if(equalsChanges[1].startsWith(plus)){
									String numberString = equalsChanges[1].substring(1);
									try{
										//make sure the number can be converted to an int
										int number = Integer.parseInt(numberString);
										variablesAndValueChanges.put(string, Integer.toString(number));
									}catch(NumberFormatException e){ //broken
										logger.severe("number is not able to be converted from String");

										return false;
									}
								}if(equalsChanges[1].startsWith(minus)){
									String numberString = equalsChanges[1].substring(1);
									try{
										//make sure the number can be converted to an int
										int number = Integer.parseInt(numberString);
										variablesAndValueChanges.put(string, "-"+Integer.toString(number));
									}catch(NumberFormatException e){ //broken
										logger.severe("number is not able to be converted from String");

										return false;
									}

								}else{ // we only support plus and minus operations so the template must be broken
									logger.severe("minus or plus not provided");
									//									return false;
								}
							}else{//template is broken
								logger.severe("equal changes not correct format");
								return false;
							}

						}
					}
				}

				hasInstruction = true;


			}
		}


		return hasInstruction; //if we get here we know everything is ok
	}

	private boolean confirmBeginTemplateInstruction(){
		for(int i = 0; i<templateArray.length; i++){
			logger.info(i+": " +templateArray[i]);
		}
		String begin = templateArray[templateArray.length-2].trim();
		String mark = templateArray[templateArray.length-1].trim();
//		logger.info("begin: ");
//		logger.info(begin);
		if(!(begin.equalsIgnoreCase(beginTemplateInstruction)) && (mark.startsWith("<"))){
			return false; //template is broken
		}


		return true; //if we get here we know everything is ok
	}

	public boolean templateOK(){
		return templateOK;
	}

	public String getFullTemplate(){
		return fullTemplate;
	}

	public String getError(){
		return errorString;
	}

	private void setCurrentValue(String key, int value, int max){
		int[] update={value, max};
		variablesAndValues.put(key, update);
	}



	/*
	 * just for local testing
	 */
	public static void main(String[] args){
		String template = "var:%i=0to47;var:%j=0;repeat template using:%i++;begin template;<row%icol%j><Station>row%icol%j+1</Station><Current_Condition>row%icol%j+2</Current_Condition><Temperature_Degrees_Centegrade>row%icol%j+3</Temperature_Degrees_Centegrade><Dew_Point_Degrees_Centegrade>row%icol%j+4</Dew_Point_Degrees_Centegrade><Humidity_Percentage>row%icol%j+5</Humidity_Percentage><Wind_Average_KMH>row%icol%j+6</Wind_Average_KMH><Wind_Gust_KMH>row%icol%j+7</Wind_Gust_KMH><Rain_MM>row%icol%j+8</Rain_MM><Barometer_MB>row%icol%j+9</Barometer_MB><Barometer_Trend>row%icol%j+10</Barometer_Trend><TimeStamp>row%icol%j+11</TimeStamp></row%icol%j>";
		TemplateTranslator test = new TemplateTranslator();
		test.translateTemplate(template);
		logger.info(test.getFullTemplate());

	}
}
