package ie.ucd.sixth.core.cyber.sensor;

import ie.ucd.sixth.core.SensorStatus;
import ie.ucd.sixth.core.cyber.adaptor.sensor.AbstractCyberAdaptor;
import ie.ucd.sixth.core.cyber.adaptor.sensor.CyberSensorData;
import ie.ucd.sixth.core.cyber.adaptor.sensor.SensorStream;
import ie.ucd.sixth.core.cyber.utils.geocode.BoundingBox;
import ie.ucd.sixth.core.cyber.utils.geocode.Geocoder;
import ie.ucd.sixth.core.cyber.utils.yahoo.BoundingBoxManager;
import ie.ucd.sixth.core.sensor.NodeDescription;
import ie.ucd.sixth.core.sensor.NodeProperty;
import ie.ucd.sixth.core.sensor.SensorNode;
import ie.ucd.sixth.core.sensor.data.SensorData;
import ie.ucd.sixth.core.service.IPipeService;
import ie.ucd.sixth.core.utils.Property;
import ie.ucd.sixth.core.utils.SensorSpecification;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;


public class CyberSensor extends SensorNode{
	private static final Logger log = Logger.getLogger(CyberSensor.class.getName());

	static {
		log.setLevel(Level.OFF);
	}
	
	/**
	 * important data paths that are extracted from model
	 */
	private static String entityPath = "";
	private static String locationNamePath = "";
	private static String latitudePath = "";
	private static String longitudePath = "";
	private static String userIdPath = "";
	private static String userNamePath = "";

	private static final long serialVersionUID = 1L;
	public static final String STATEPROPERTY = "state"; //every sensor must have a state that can be set to on or off
	public static final String FREQUENCYPROPERTY = "frequency"; //every sensor must have a frequency that either determines how often to sample or monitors how many responses an API stream receive per unit of time
	public static final String PIPES_PROPERTY = "pipe";
	public static final String NETWORKIDPROPERTY = "networkid";
	public static final String INFRASTRUCTUREIDPROPERTY = "infrastructureid";
	protected boolean active = false; //every sensor must maintain a boolean indicating whether it is active or not
	protected AbstractCyberAdaptor adaptor; //every sensor must have reference to the adaptor from which it was created
	protected SensorStream sensorStream;
	protected HashMap<String, Property> setPropertiesMap;

	protected List<IPipeService> pipeList;
	private String sensorName;
	private Map<String, String> sensorMap;
	private List<Property> sensorProperties;
	private int networkID; //for use in network abstractions
	private int infrastructureID; //for use in infrastructure abstractions
	private String sensedModality="";
	private int sensorFrequency = 1;
	/*
	 * create a new cybersensor 
	 */
	
	public CyberSensor(AbstractCyberAdaptor adaptor, int id, String sensorType, SensorSpecification spec){
		super(new NodeDescription(id, adaptor.getType()), adaptor, sensorType);
		this.adaptor = adaptor;
		this.active = false;
		this.sensorStream = new SensorStream(this, sensorType);
		setPropertiesMap = new HashMap<String, Property>();
		pipeList = new ArrayList<IPipeService>();
		sensorName = spec.getName();
		sensorMap = spec.getMap();
		sensorProperties = spec.getProperties();
//		System.out.println("sensor specification in cyber sensor: " +spec.toString());
		for (Property sensorProperty : sensorProperties) {
			setPropertiesMap.put(sensorProperty.getName(), sensorProperty);
//			System.out.println("cyber sensor added property: "+sensorProperty.getName()+ " which has type: " +sensorProperty.getType() );
		}
	}

	public String getSensedModality(){
		return sensedModality;
	}
	/*
	 * method for the adaptor to use when setting a sensor property
	 * set  a value for one of the predefined properties for this sensor
	 * if the property is state the sensor stream should be turned on or off accordingly
	 * when a new property is set the updateSetPropertiesMap method should be called to reflect the changes
	 * 
	 * this method should handle any configuration errors caused by the users message
	 * (see setsenorproperty above)
	 */
	public void setSensorProperty(String propertyName, String value){
		updateSetPropertiesMap(propertyName, value);
	}



	/*
	 * when a property is set we need to update the map
	 */
	protected void updateSetPropertiesMap(String propertyName, String value){
		if(value!=null &&!value.isEmpty() &&!value.equalsIgnoreCase("null")){
			if(setPropertiesMap.containsKey(propertyName)){
				if(propertyName.equalsIgnoreCase(NETWORKIDPROPERTY)){
					
					this.networkID = Integer.parseInt(value);
					properties.put("networkid", new NodeProperty(value));
				}
				if(propertyName.equalsIgnoreCase(INFRASTRUCTUREIDPROPERTY)){
					this.infrastructureID = Integer.parseInt(value);
					properties.put("infrastructureid", new NodeProperty(value));
				}
				//handle sensorFrequency
				if(propertyName.equalsIgnoreCase(FREQUENCYPROPERTY)){
					this.sensorFrequency = Integer.parseInt(value);
				}
				if(propertyName.equalsIgnoreCase(STATEPROPERTY)){
					if(value.equalsIgnoreCase("on")){
						stream(sensorFrequency);
						this.active = true;
					}
				}
				if(propertyName.equalsIgnoreCase(PIPES_PROPERTY)){					
					String[] pipeArr = value.split(";");
					int[] pipes = new int[pipeArr.length];
					int i =0;
					for (String pipeString : pipeArr) {
						int pipeid = Integer.parseInt(pipeString);
						pipes[i] = pipeid;
						i++;
					}
			
					pipeList = adaptor.getPipeList(pipes);
				

				}
				//handle locations here to make sure a sensor has enough variations to suit most api requirements
				if(propertyName.equalsIgnoreCase("locationname")){
					sensedModality = value;
					//get the location value for latlong, boundingbox
					Map<String, String> locationMap = Geocoder.geocode(value);
					String boundingboxpath = "searchresults.place.0.boundingbox";
					String latitudepath = "searchresults.place.0.lat";
					String longitudepath = "searchresults.place.0.lon";

					String boundingboxString = locationMap.get(boundingboxpath);
					Property bbProp = new Property(setPropertiesMap.get("boundingbox").getMap(), boundingboxString);
//					System.out.println("cyber sensor has set bounding box property: " +boundingboxString);
//					System.out.println("bbrprop in cybersensor: "+bbProp.getType()+" "+bbProp.getContent());
					setPropertiesMap.put("boundingbox", bbProp);
					String latString = locationMap.get(latitudepath);
					String lonString = locationMap.get(longitudepath);
					Property latlonProp = new Property(setPropertiesMap.get("latlong").getMap(), latString+","+lonString);
					setPropertiesMap.put("latlong", latlonProp);
				}
				else if(propertyName.equalsIgnoreCase("latlong")){
					sensedModality=value;
					//get the location value for locationname, boundingbox
					String[] vals = value.split(";");
					if(vals!=null && vals.length==2){
						Map<String, String> locationMap = Geocoder.reverseGeocode(vals[0], vals[1]);
						String countyPath = "reversegeocode.addressparts.county";
						String locationName = locationMap.get(countyPath);
						Property nameProp = new Property(setPropertiesMap.get("locationname").getMap(), locationName);
						setPropertiesMap.put("locationname", nameProp);
						//need to make another call to get a bounding box value
						Map<String, String> newLocationMap = Geocoder.geocode(locationName);
						String boundingboxpath = "searchresults.place.0.boundingbox";
						String boundingboxString = newLocationMap.get(boundingboxpath);
						String[] bbarr = boundingboxString.split(",");
						Property bbProp = new Property(setPropertiesMap.get("boundingbox").getMap(), boundingboxString);
						setPropertiesMap.put("boundingbox", bbProp);
					}
				}
				else if(propertyName.equalsIgnoreCase("boundingbox")){
					sensedModality = value;
					String[] vals = value.split(";");
					if(vals!=null && vals.length==4){
					
//						Map<String, String> locationMap = Geocoder.reverseGeocode(vals[0], vals[1]);
//						String countyPath = "reversegeocode.addressparts.county";
//						String locationName = locationMap.get(countyPath);
//						Property nameProp = new Property(setPropertiesMap.get("locationname").getMap(), locationName);
//						setPropertiesMap.put("locationname", nameProp);
//						//need to make another call to get a bounding box value
//						Map<String, String> newLocationMap = Geocoder.geocode(locationName);
//						String boundingboxpath = "searchresults.place.0.boundingbox";
//						String boundingboxString = newLocationMap.get(boundingboxpath);
//						String[] bbarr = boundingboxString.split(",");
						Property bbProp = new Property(setPropertiesMap.get("boundingbox").getMap(), sensedModality);
						setPropertiesMap.put("boundingbox", bbProp);
					}
				}
		
				else if(propertyName.equalsIgnoreCase("keyword")){
					sensedModality=value;
				}
				else if(propertyName.equalsIgnoreCase("userid")){
					sensedModality=value;
				}
				Property property = new Property(setPropertiesMap.get(propertyName).getMap(), value);
				setPropertiesMap.put(propertyName, property);
			}
		}
	}


	/*
	 * check if a property has a value matching the value provided
	 */
	public boolean checkProperties(String propertyName, String value){
		if(setPropertiesMap.containsKey(propertyName)){
			if((setPropertiesMap.get(propertyName).getContent()).equalsIgnoreCase(value)){
				return true;
			}
		}
		return false;
		//		Set<String> keyset = setPropertiesMap.keySet();
		//		if(keyset.contains(propertyName)){
		//			String propertyValue = setPropertiesMap.get(propertyName);
		//			//sometimes values are added as a sequence of ";" separated strings so we need to check them all
		//			String[] valueArray = propertyValue.split(";");
		//			for (String string : valueArray) {
		//				if((string.trim()).equalsIgnoreCase((value.trim()))){ 
		//					return true; //if they match we can return true
		//				}
		//			}
		//		}
		//		return false; //return false to indicate nothing matched
	}

	/*
	 * get the properties map for this sensor
	 */
	public HashMap<String, Property> getSetPropertiesMap(){
		return setPropertiesMap;
	}

	/*
	 * provide functionality that will allow a sensor to queue sensordata objects in a stream
	 */
	public void queueData(CyberSensorData data){
		if (!data.getValues().isEmpty()) {
			this.sensorStream.addSensorData(data);
		}
	}

	/*
	 * send data to the receiver 
	 */
	public void sendData(SensorData data) {
		log.info("abstractcybersensor.sendData: " +data.getSensedModality());
		getAdaptor().addData(data);
	}

	public void sendSensorCreationConfirmation(int sensorId, String sensorType){
		String dataString = "<sixth><notification type='sensor created'><sensorType>"+sensorType+"</sensorType><sensorId>"+sensorId+"</sensorId></notification></sixth>";

		Map<String, String> map = new HashMap<String, String>();
		map.put("sensorType", sensorType); 
		map.put("sensorId", sensorId+"");

		SensorData data = new SensorData(this.getNetwork(), sensorId, "sixth.notification", map, SensorStatus.NEW.toString(), System.currentTimeMillis());
		sendData(data);
	}

	/*
	 * get the stream for this sensor
	 */
	public SensorStream getSensorStream(){
		return this.sensorStream;
	}

	/*
	 * start streaming data from this sensor
	 */

	public void stream(int frequencyInSeconds){
		sensorStream.stream(frequencyInSeconds);
	}

	/*
	 * check if this sensor is active
	 */
	public boolean isActive(){
		return  active;
	}


	/*
	 * retrieve this sensors current list of pipes
	 */
	public synchronized List<IPipeService> getPipes(){
		return pipeList;
	}

	/*
	 * set the pipeList
	 * TODO: for now I am doing this by list
	 * in future versions we should allow pipes to be inserted individually and in a specified order
	 */
	public synchronized void setPipeList(List<IPipeService> pipes){
		this.pipeList = pipes;
	}

	public synchronized List<Property> getProperties() {
		return sensorProperties;
	}

	public boolean checkBoundingBox(Double latitude, Double longitude) {
		if(setPropertiesMap.containsKey("boundingbox")){
			Property property = setPropertiesMap.get("boundingbox");
			String boundingBoxValue = property.getContent();
			if(boundingBoxValue!=null && !boundingBoxValue.isEmpty()){
				if(boundingBoxValue.equalsIgnoreCase("-180,-90,180,90")){
					return true;
				}
				//there may be multiple bounding boxes seperated by '';'
				String[] boundingboxes = boundingBoxValue.split(";");
				for (String bbox : boundingboxes) {
					BoundingBox boundingBox = new BoundingBox(bbox);
					return boundingBox.contains(latitude, longitude);			
				}
			}
		}
		return false;
	}

	public int getNetworkID(){
		return networkID;
	}
}
