package ie.ucd.sixth.core.cyber.adaptor.sensor;

import ie.ucd.sixth.core.cyber.sensor.CyberSensor;
import ie.ucd.sixth.core.sensor.data.ISensorData;
import ie.ucd.sixth.core.sensor.data.SensorData;
import ie.ucd.sixth.core.service.IPipeService;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * Standard stream implementation to be used for all cyber sensors
 * The frequency here dictates how often the sensor will send its readings to the client
 * Readings from a relevant cyberAdaptorStream should be queued for sending using the addSensorData(SensorData data) method.
 * The sensorPushFrequency for the sensor that owns this thread must be set
 * If there is no data waiting to be sent the stream will wait until the next interval to try again
 * 
 * each piece of data to be sent is pushed through all pipes applied to the sensor it came from before being sent to the application
 * 
 * @author Olga Murdoch
 */
public class SensorStream implements Runnable {
	//debugging
	private boolean haveSentData = false;



	protected static Logger logger = Logger.getLogger(SensorStream.class.getName());

	static {
		logger.setLevel(Level.OFF);
	}
	//provide variables to store properties such as locations and keywords

	//every stream needs a frequency (and this can be configured too)
	private int frequencySeconds;

	//give the stream access to the sensor objects that can be used to retrieve the information

	//keep track of whether the stream is on or off
	private boolean threadStopped;
	//make note of the type of sensor we are working with
	private String sensorType = "";
	//variable to store the sensor object the stream is connected to
	private CyberSensor sensor;

	//List of sensor data objects that have been queued to send at the frequency dictated by the stream
	private ArrayList<CyberSensorData> dataList;

	private Thread thread;

	private Boolean allData = false; //allow all queued data to be set if this value is set to true. Default value is false;
	/*
	 * constructor - stream needs access to the sensor being monitored and its type
	 */
	public SensorStream(CyberSensor sensor, String sensorType){
		this.sensor = sensor;
		this.sensorType = sensorType;
		//the thread must start off as being off
		threadStopped = true;

		//initialize the dataList
		dataList = new ArrayList<CyberSensorData>();

	}

	/*
	 * constructor - stream needs access to the sensor being monitored and its type
	 */
	public SensorStream(CyberSensor sensor, String sensorType, Boolean allData){
		this.sensor = sensor;
		this.sensorType = sensorType;
		//the thread must start off as being off
		threadStopped = true;

		//initialize the dataList
		dataList = new ArrayList<CyberSensorData>();
		this.allData = allData;

	}



	/*
	 * default stream method
	 * @param frequencyInSeconds all streams require the frequency to be set
	 */
	public void stream(int frequencyInSeconds){
		logger.info("starting stream with frequency: " +frequencyInSeconds);
		//make sure the thread starts off as stopped
		threadStopped = true;
		//		updateFrequency(0); 

		//reset the frequency
		updateFrequency(frequencyInSeconds);


		//once all property values have been set we can start the thread
		threadStopped = false;
		this.thread = new Thread(this);
		thread.start();
	}




	/*
	 * send all of the queued data for this sensor 
	 * SensorData data = new SensorData(sensor.getId(), sensorType, dataToSend, SIXTH.TYPE_PERIODIC);
	 * sensor.sendData(data);
	 */
	public void sendData(ArrayList<CyberSensorData> queuedSensorData){

		List<CyberSensorData> localList = queuedSensorData;



		logger.info("in send data method....");

		//		logger.info("size of queued sensor data in send method: " +localList.size());
		List<IPipeService> pipeList = sensor.getPipes();
		//		logger.info("got this many pipes from sensor: " +pipeList.size());
		//		logger.info("sending data**************************************************************");

		for (CyberSensorData sensorData : localList) {
			//			logger.info("in for loop");
			if(sensorData!=null){
				//				logger.info("sensor data is not null");
				if(pipeList!=null && !pipeList.isEmpty()){
					logger.info("APPLYING PIPES");

					sensorData = applyPipes(sensorData);
					if(sensorData!=null){
						haveSentData = true;
						sensor.sendData(sensorData);						
					}
				}else{
					logger.info("not applying pipes");


					haveSentData = true;
					sensor.sendData(sensorData);						
				}

			}else{
				logger.info("can not send null sensorData");
			}
		}

	}
	/*
	 * standard run method
	 * requests should me made every X number of seconds where X=frequency configured in sensor
	 */
	@Override
	public void run() {
		while(getFrequency()>0){
			threadStopped = false;

			ArrayList<CyberSensorData> localDataList = getSensorDataList();
			logger.info("in thread... size of list: " +localDataList.size()+"......................... " +sensor.getID()+": " +sensor.getSensorType());
			logger.info("***************** double checking if list is empty: " +localDataList.isEmpty());
			if(!localDataList.isEmpty()){
				logger.info("calling send data method from run thread...."+"......................... " +sensor.getID()+": " +sensor.getSensorType() +" number of sensor data to send = " +localDataList.size());
				sendData(localDataList);	

			}else{
				logger.info("there is no data queued to be sent"+"......................... " +sensor.getID()+": " +sensor.getSensorType());
			}

			try {
				Thread.sleep((getFrequency()*1000)); //convert seconds to milliseconds and wait that long before next request
			} catch (InterruptedException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}

		}

		threadStopped = true;

	}


	/*
	 * returns the current frequency for this stream
	 */
	public synchronized int getFrequency() {
		if(frequencySeconds==0 && haveSentData){
			System.exit(0);
		}
		return frequencySeconds;
	}

	/*
	 * returns the current data list to be sent by this stream
	 */
	public synchronized ArrayList<CyberSensorData> getSensorDataList(){		
		ArrayList<CyberSensorData> temporary;
		logger.info("getting dataList which currently has size: " +this.dataList.size());
		synchronized (dataList){
			temporary = new ArrayList<CyberSensorData>(dataList);
			dataList.clear();
			//			logger.info("returning temp.. it has size: " +temporary.size());
			//			logger.info("dataList has been reset to size: " +this.dataList.size());
		}
		return temporary;
	}

	/*
	 * change the current frequency for this stream
	 */
	public synchronized void updateFrequency(int frequency) {
		logger.info("updating sensor stream frequency to: " +frequency );
		this.frequencySeconds = frequency;

		if ((frequency > 0) && threadStopped) {
			new Thread(this).start();
		}

	}

	/*
	 * add a sensordata to the queue list
	 */
	public synchronized void addSensorData(CyberSensorData data){
		logger.info("adding sensor data to list for this sensor" +sensor.getNetwork()+"."+sensor.getSensorType()+"."+sensor.getID());
		//		System.out.println("sensor is stopped? " +threadStopped);
		synchronized (dataList){
			//			if(!allData){
			//				dataList = new ArrayList<SensorData>();
			//			}
			if(data!=null){
				dataList.add(data);				

				logger.info("datalist size after adding: " +dataList.size());				
			}

		}
		logger.info("is this sensor streams thread stopped? " +threadStopped);
		logger.info("what is the frequency for this sensor stream? " +frequencySeconds);
	}


	public synchronized CyberSensorData applyPipes(CyberSensorData data){
		logger.info("in sensorStream.applypipes() ... is data null? " +(data==null));
		List<IPipeService> pipeList = sensor.getPipes();
		logger.info("have this many pipes to apply to sensor data: " +pipeList.size());
		for (IPipeService iPipeService : pipeList) {
			logger.info("deal with this pipe?? " +(iPipeService==null));
			logger.info(iPipeService.getName()+" "+iPipeService.getType() +" " +iPipeService.getID());
			if(sensor.getProperty("networkid")!=null){//if the sensor is the base sensor for a network abstraction add the network id to the data for sensor producer pipes
				Map<String, String> newValues = data.getValues();
				newValues.put("networkid", sensor.getProperty("networkid").getValue());
				data.replaceValues(newValues);
			}else{
				Map<String, String> newValues = data.getValues();
				newValues.put("networkid", sensor.getNetworkID()+"");
				data.replaceValues(newValues);
			}
			logger.info("is my pipe null? "+(iPipeService==null));
			data = (CyberSensorData) iPipeService.pipeData(data);


			if (data != null) { //repackage the data to include the new sensedmodality
				CyberSensorData newData = new CyberSensorData(data.getNetwork(), data.getID(), data.getSensedModality()+":"+iPipeService.getName(), data.getValues(), ISensorData.TYPE_PERIODIC, data.getTimestamp(), data.getDataModel());			
				data = newData;
			}else{
				return null;//if the data was filtered out we return null to say we dont want to send it

			}
			
			
		}

		return data; 	
	}

}
