package ie.ucd.sixth.core.cyber.adaptor.sensor;

import ie.ucd.sixth.core.cyber.sensor.CyberSensor;
import ie.ucd.sixth.core.utils.Property;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * The AbstractAdaptorStream should be extended for use in any cyber sensing adaptor that needs to interface with a Streaming API
 * The stream adaptor aggregates all sensors and properties, keeping track of how many sensors are interested in a particular value 
 * This should be used in future extensions to support QoS requirements 
 * @author olgamurdoch1_old_iMac
 *
 */
public abstract class AbstractAdaptorStream {

	protected static Logger logger = Logger.getLogger(AbstractAdaptorStream.class.getName());
	
	static {
		logger.setLevel(Level.OFF);
	}

	protected Map<String, AggregateProperty> aggregateProperties;
	protected HashMap<Integer, CyberSensor> sensorMap;
	protected AbstractCyberAdaptor adaptor;
	protected boolean active = false;
	private int maxCallsPerSecond;
	

	public AbstractAdaptorStream(AbstractCyberAdaptor adaptor, int maxCallsPerSecond){
		sensorMap = new HashMap<Integer, CyberSensor>();
		this.adaptor = adaptor;
		aggregateProperties = new HashMap<String, AggregateProperty>();
		this.maxCallsPerSecond = maxCallsPerSecond;
	}
	
	public AbstractAdaptorStream(AbstractCyberAdaptor adaptor){
		sensorMap = new HashMap<Integer, CyberSensor>();
		this.adaptor = adaptor;
		aggregateProperties = new HashMap<String, AggregateProperty>();

	}


	/*
	 * provide functionality to stream data from the source to the relevant sensors
	 */	
	public abstract void stream();

	/*
	 * provide functionality to shut down the stream
	 * active should be set to false to reflect the change in this streams status
	 */
	public abstract void shutdown();

	/*
	 * provide functionality to start the stream
	 * active should be set to true to reflect the change in this streams status
	 */
	public abstract void start();

	/*
	 *add new sensor properties to those being monitored by this stream
	 *NB: this method is to be used by addSensor(sensor) only
	 */
	protected synchronized void addSensorProperties(CyberSensor sensor) {
		logger.info("adding sensor properties to map: " +sensor.getSensorType()+":"+sensor.getID());
		HashMap<String, Property> sensorProperties = sensor.getSetPropertiesMap();
		for (String string : sensorProperties.keySet()) {
			System.out.println("[abstract adaptor stream ] key: " +string+ "    values: " +sensorProperties.get(string).getName()+", " +sensorProperties.get(string).getType()+", "+sensorProperties.get(string).getContent());
		}
		boolean hasProperties = false;
		boolean anyChange=false;
		if(sensorProperties.isEmpty()){
			return;
		}

		Set<String> properties = sensorProperties.keySet();
		for (String propertyName : properties) {
			logger.info("trying for property: "+propertyName+"=" +sensorProperties.get(propertyName));
			System.out.println("what im adding .... "+sensorProperties.get(propertyName).getContent()+" ............... for sensor id: " +sensor.getID() +"and property "+propertyName);
			if(aggregateProperties.containsKey(propertyName)){
				//this property already has values so we need to update the aggregateproperty
				hasProperties = aggregateProperties.get(propertyName).addValue(sensorProperties.get(propertyName).getContent(), sensor.getID());
				if(hasProperties){
					anyChange=true;
				}
			}else{
				//it hasn't been set so we create a new aggregateProperty
				AggregateProperty aggProp = new AggregateProperty(propertyName, sensorProperties.get(propertyName));
				aggProp.addValue(sensorProperties.get(propertyName).getContent(), sensor.getID());
				aggregateProperties.put(propertyName, aggProp);
				anyChange=true;
				
			}
		}
		
		if(anyChange){ //we only want to restart the stream if the aggregate properties map has been updated
			logger.severe("HAVE NEW PROPERTIES AND NEED TO RESTART STREAM");
			queueStart(sensor);
		}else{
			logger.severe("NO NEW PROPERTIES HERE..... HASHTAG OBLIVIOUS");
		}
	}

	protected synchronized Set<String> getPropertySet(){
		return aggregateProperties.keySet();
	}

	protected synchronized AggregateProperty getAggregateValues(String propertyName){
		return aggregateProperties.get(propertyName);
	}



	/*
	 * allows a new sensor to be added to the adaptor stream for this type of sensor
	 * NB: this method restarts the stream so a new sensor should only be added when it is acceptable to do this
	 *  - to add multiple sensors at a time use the addSensorList method
	 */
	public synchronized void addSensor(CyberSensor sensor){
		logger.info("adding sensor to adaptor stream: "+sensor.getID()+", "+sensor.getNetwork()+", "+sensor.getSensorType()+", "+sensor.getSensedModality());
		//TODO: for adding sensor to adaptor stream and checking if sensor properties need to be updated
		if(sensor!=null){
			int id = sensor.getID();
			if(!sensorMap.containsKey(id)){ //if the sensor is not in the map we should add it
				addSensorProperties(sensor);
				sensorMap.put(id, sensor);
			}else{//if the sensor is already in the map we need to check if the sensor properties need to be updated
				addSensorProperties(sensor);
			}
	
		}

	}
	
	
	/*
	 * allow properties for a specific sensor to be updated
	 * if this sensor does not already exist in the map it will be added
	 */
	public synchronized void updateSensorProperties(CyberSensor sensor){
		int id = sensor.getID();
		if(!sensorMap.containsKey(id)){
			addSensor(sensor);
		}else{
			addSensorProperties(sensor);
			
		}
	}

	/*
	 * allows a lsit of new sensors to be added to the adaptor stream for this type of sensor
	 * NB: this method restarts the stream so a new sensor should only be added when it is acceptable to do this
	 */
	public synchronized void addSensorList(List<CyberSensor> sensors){
		for (CyberSensor sensor : sensors) {
			if(sensor!=null){
				int id = sensor.getID();
				if(!sensorMap.containsKey(id)){ //we only want to add new sensors
					addSensorProperties(sensor);
					sensorMap.put(id, sensor);
				}
			}
		}

		shutdown();
		System.out.println("ADDED NEW LIST OF SENSORS.... SO GOING TO START ADAPTOR STREAM. ");
		start();
	}

	/*
	 * send a sensordata object to a relevant sensor
	 */
	public void pushDataToSensor(CyberSensor sensor, CyberSensorData data){
		//logger.info("pushing data to sensor... " +sensor.getID()+" "+data.getValues());
		if(data!=null){
			sensor.queueData(data);			
		}
	}

	/*
	 * check if this adaptor stream is active
	 */
	public boolean isActive(){
		return active;
	}


	public synchronized boolean allSensorsOff(){
		for (Entry<Integer, CyberSensor> entry : sensorMap.entrySet()) {
			CyberSensor sensor = entry.getValue();
			if(sensor.isActive()){
				return false;
			}
		}
		return true;
	}

	public Set<String> getValuesForProperty(String propertyName){
		if(aggregateProperties.containsKey(propertyName)){
			return aggregateProperties.get(propertyName).getValues();
		}
		return null;
	}
	
	public List<Integer> getSensorsForValue(String propertyName, String value){
		if(aggregateProperties.containsKey(propertyName)){
			AggregateProperty prop = aggregateProperties.get(propertyName);
			return prop.getSensorsForValue(value);
		}
		return null;
	}


	protected abstract void queueStart(CyberSensor sensor);
}
